/* 
 *  Logger.java
 * 
 *  Copyright 2016 Avaya Inc. All Rights Reserved.
 * 
 *  Usage of this source is bound to the terms described
 *  in AvayaLicenseSDK.rtf.
 * 
 *  Avaya - Confidential & Proprietary. Use pursuant to your signed agreement
 *  or Avaya Policy
 * 
 */
package com.avaya.ccs.javafxrefclient;

import com.avaya.ccs.api.ErrorI;
import java.time.Duration;
import org.apache.logging.log4j.LogManager;

/**
 * This class forms the entry point for logging within the reference client The
 * logging should appear in the same log file as the sdk logs themselves.
 */
public class Logger {

    private static final int DISPLAY_ERROR_FOR_SECONDS = 10;
    Class<?> clazz;
    org.apache.logging.log4j.Logger log;
    long mainErrorTime = 0;

    private Logger(Class<?> clazz) {
        this.clazz = clazz;
        this.log = LogManager.getLogger(clazz);
    }

    public static Logger getLogger(Class<?> clazz) {
        return new Logger(clazz);
    }

    public void trace(String msg, Object... args) {
        clearStatusIfTimeout();
        log.trace(msg, args);
    }

    public void info(String msg, Object... args) {
        clearStatusIfTimeout();
        log.info(msg, args);
    }

    public void debug(String msg, Object... args) {
        clearStatusIfTimeout();
        log.debug(msg, args);
    }

    public void warn(String msg, Object... args) {
        clearStatusIfTimeout();
        log.warn(msg, args);
    }

    public void error(String methodName, Throwable t, String msg) {
        clearStatusIfTimeout();
        log.error(methodName + "() " + msg, t);
    }

    public void error(String methodName, Throwable t) {
        clearStatusIfTimeout();
        log.error(methodName + "() ", t);
    }

    public void error(String msg, Object... args) {
        clearStatusIfTimeout();
        log.error(msg, args);
    }

    public void errorStatus(ErrorI error) {
        String msg = error.getCommand() + " error - " + error.getMessage();
        
        mainErrorTime = System.currentTimeMillis();
        JavaFXRefClient.UpdateStatus(msg);

        log.error("errorStatus():" + msg);
    }

    public void errorStatus(String msg) {
        mainErrorTime = System.currentTimeMillis();
        JavaFXRefClient.UpdateStatus("System error: " + msg);

        log.error("errorStatus():" + msg);
    }

    public void infoStatus(String msg) {
        mainErrorTime = System.currentTimeMillis();
        JavaFXRefClient.UpdateStatus("System info: " + msg);

        log.info("infoStatus():" + msg);
    }
    
    public void clearStatus() {
        JavaFXRefClient.UpdateStatus("");
        mainErrorTime = 0;
    }

    public void clearStatusIfTimeout() {
        long currentTime = System.currentTimeMillis();
        
        if(mainErrorTime != 0){
            if((currentTime - mainErrorTime) > Duration.ofSeconds(DISPLAY_ERROR_FOR_SECONDS).toMillis() ){
                JavaFXRefClient.UpdateStatus("");
                mainErrorTime = 0;
            }
        }
    }
}
